"""LMS-31D用シリアル制御モジュール
"""

# ******************************************************************************
# (c) 2023 LAND COMPUTER Co.,Ltd.
# SPDX-License-Identifier: BSD-3-Clause
#
# LMS-31D製品ページ：https://www.landcomp.co.jp/product/peripheral/hdmi/lms-31d/
#
# [2023/06/01] Ver.1.0.0.0
# ******************************************************************************

import serial
from collections import namedtuple

# ******************************************************************************
# 定数
# ******************************************************************************
TIMEOUT_SEC_DEFAULT_READ: float = 1
TIMEOUT_SEC_DEFAULT_WRITE: float = 1

CMD_HEADER: bytes = b"\x1b"
CMD_TERMINATOR: bytes = b"\x0d"

CMD_FLAG_POWER: str = "PWR"
CMD_FLAG_SOURCE: str = "SRC"
CMD_FLAG_GET: str = "GET"

SOURCE_NUM_MIN: int = 0  # 未選択
SOURCE_NUM_MAX: int = 3

# ******************************************************************************
# 型定義
# ******************************************************************************
result_power_state = namedtuple("result_power_state", ["result", "power_state"])
result_source_state = namedtuple("result_source_state", ["result", "source_state"])


# ******************************************************************************
# クラス実装
# ******************************************************************************
class SerialForLMS31D:
    """LMS-31D用シリアル通信クラス"""

    # --------------------------------------------------------------------------
    # Private
    # --------------------------------------------------------------------------

    def __init__(self) -> None:
        self.__serial = None

    def __del__(self) -> None:
        self.close()

    def __read_text(self, a_timeout_sec: float) -> str:
        """文字列読み取り

        Args:
            a_timeoutsec (float): タイムアウト値（秒）

        Returns:
            str: 文字列
        """

        # タイムアウト値設定
        self.__serial.timeout = a_timeout_sec

        # ターミネーター（0x0D）待ち読み取り
        __data: bytes = self.__serial.read_until(CMD_TERMINATOR)

        # エンコードに影響あるため先にヘッダー・ターミネーター削除
        __data = __data.replace(CMD_HEADER, b"")
        __data = __data.replace(CMD_TERMINATOR, b"")

        # 文字列化
        return __data.decode("utf-8")

    def __write_text(self, a_text: str) -> bool:
        """文字列書き込み

        Args:
            a_text (str): 文字列

        Returns:
            bool: 書き込み成功可否
        """

        try:
            # 書き込み
            self.__serial.write(CMD_HEADER + a_text.encode("utf-8") + CMD_TERMINATOR)

            # 書き込み待ち
            self.__serial.flush()

            return True

        except serial.SerialTimeoutException:
            return False

    def __receive_power(self) -> result_power_state:
        """電源状態受信

        Returns:
            result_power_state: result=成功可否 power_state=電源状態
        """

        # 読み込み
        __text: str = self.__read_text(TIMEOUT_SEC_DEFAULT_READ)

        # データ確認
        if (len(__text) > 0) & (__text.startswith(CMD_FLAG_POWER.lower())):
            # フラグ削除
            __text = __text.replace(CMD_FLAG_POWER.lower(), "")

            # 整数確認
            if __text.isdecimal():
                # 状態取得
                __state: int = int(__text)

                # 状態範囲確認
                if (__state == 0) | (__state == 1):
                    return result_power_state(
                        result=True, power_state=(True if __state == 1 else False)
                    )

        return result_power_state(result=False, power_state=False)

    def __receive_source(self) -> result_source_state:
        """電源状態受信

        Returns:
            result_source_state: result=成功可否 power_state=ソース選択状態
        """

        # 読み込み
        __text: str = self.__read_text(TIMEOUT_SEC_DEFAULT_READ)

        # データ確認
        if (len(__text) > 0) & (__text.startswith(CMD_FLAG_GET.lower())):
            # フラグ削除
            __text = __text.replace(CMD_FLAG_GET.lower(), "")

            # 整数確認
            if __text.isdecimal():
                # 状態取得
                __state: int = int(__text)

                # 状態範囲確認
                if (__state >= SOURCE_NUM_MIN) & (__state <= SOURCE_NUM_MAX):
                    return result_source_state(result=True, source_state=__state)

        return result_source_state(result=False, source_state=SOURCE_NUM_MIN)

    # --------------------------------------------------------------------------
    # Public
    # --------------------------------------------------------------------------

    def try_open(self, a_port: str) -> bool:
        """接続トライ

        Args:
            a_port (str): Win="COM3" Linux="/dev/ttyAMA0" 等

        Returns:
            bool: 成功可否
        """
        try:
            # シリアル通信初期化
            self.__serial = serial.Serial(
                port=a_port,
                baudrate=9600,
                bytesize=serial.EIGHTBITS,
                rtscts=True,
                xonxoff=False,
                parity=serial.PARITY_NONE,
                stopbits=serial.STOPBITS_ONE,
                timeout=TIMEOUT_SEC_DEFAULT_READ,
                write_timeout=TIMEOUT_SEC_DEFAULT_WRITE,
            )

            # バッファクリア
            self.__serial.reset_input_buffer()
            self.__serial.reset_output_buffer()

        except serial.SerialException:
            return False

        return True

    def close(self) -> None:
        """切断"""

        if self.__serial is not None:
            # バッファクリア
            self.__serial.reset_input_buffer()
            self.__serial.reset_output_buffer()

            # 切断
            self.__serial.close()

    def control_power(self, a_on: bool) -> bool:
        """電源制御

        Args:
            a_on (bool): 電源状態

        Returns:
            bool: 成功可否
        """

        # 書き込み
        if self.__write_text(CMD_FLAG_POWER + ("1" if a_on else "0")):
            # 状態が返ってくるので受信
            __result: result_power_state = self.__receive_power()

            # 受信成功可否
            if __result.result:
                # 照合
                return __result.power_state == a_on

        return False

    def control_source(self, a_source: int) -> bool:
        """ソース選択制御

        Args:
            a_source (int): ソース番号（0=未選択）

        Returns:
            bool: 成功可否
        """

        # 範囲確認
        if (a_source >= SOURCE_NUM_MIN) & (a_source <= SOURCE_NUM_MAX):
            # 書き込み
            return self.__write_text(CMD_FLAG_SOURCE + str(a_source))

        return False

    def request_state_power(self) -> result_power_state:
        """電源状態要求

        Returns:
            result_power_state: result=成功可否 power_state=電源状態
        """

        # 要求書き込み
        if self.__write_text(CMD_FLAG_POWER + "?"):
            # 受信
            return self.__receive_power()

        return result_power_state(result=False, power_state=False)

    def request_state_source(self) -> result_source_state:
        """ソース選択状態要求

        Returns:
            result_source_state: result=成功可否 source_state=ソース選択状態（0=未選択）
        """

        # 要求書き込み
        if self.__write_text(CMD_FLAG_GET):
            # 受信
            return self.__receive_source()

        return result_source_state(result=False, source_state=SOURCE_NUM_MIN)
